﻿#region Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// This file and its contents are protected by Romanian and International
// copyright laws. Unauthorized reproduction and/or distribution of all
// or any portion of the code contained herein is strictly prohibited
// and will result in severe civil and criminal penalties.
// Any violations of this copyright will be prosecuted
// to the fullest extent possible under law.
// ======================================================================
// THIS COPYRIGHT NOTICE MAY NOT BE REMOVED FROM THIS FILE.
// ======================================================================
#endregion

#region References
using System;
using System.Drawing;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using DataMatrix.net;
#endregion

namespace Cnas.Siui.BarCode
{
    /// <summary>
    /// Static methods to fulfill DataMatrix related operations
    /// </summary>
    public class BarcodeUtils
    {
        #region DataMatrix barcode generation
        
        /// <summary>
        /// Encodes a byte array as a Base256 DataMatrix barcode, using DataMatrix.NET library
        /// </summary>
        /// <param name="code">Byte array to encode</param>
        /// <returns>Bitmap object to store the barcode</returns>
        private static Bitmap EncodeDataMatrix( byte[] code )
        {
            var encoder = new DmtxImageEncoder();
            var options = new DmtxImageEncoderOptions
                              {
                                  ModuleSize = 2,
                                  MarginSize = 5,
                                  BackColor = Color.White,
                                  ForeColor = Color.Black,
                                  Scheme = DmtxScheme.DmtxSchemeBase256
                              };

            // calls the newly added method
            return encoder.EncodeImageBase256( code, options );        
        }

        /// <summary>
        /// Serializes a string with UTF8 encoding and Compresses it as JavaZip
        /// </summary>
        /// <param name="code">String to serialize & compress</param>
        /// <returns>Byte Array with the compressed string</returns>
        public static byte[] SerializeAndCompressString( string code )
        {
            using( var ms = new MemoryStream() )
            {
                var input = Encoding.UTF8.GetBytes(code);

                return ZipUtils.ZipArray( input, ZipUtils.ZipMethod.JavaZip );
            }   
        }

        /// <summary>
        /// Method to extract a Byte Array from a Bitmap object
        /// </summary>
        /// <param name="bmp">Source Bitmap</param>
        /// <returns>Byte Array result</returns>
        private static byte[] BmpToBytes_MemStream( Bitmap bmp )
        {
            using( var ms = new MemoryStream() )
            {
                bmp.Save( ms, System.Drawing.Imaging.ImageFormat.Bmp );
                var bmpBytes = ms.GetBuffer();
                bmp.Dispose();
                ms.Close();

                return bmpBytes;
            }
        }

        /// <summary>
        /// Generate DataMatrix barcode in a Bitmap object from a given string
        /// </summary>
        /// <param name="text">String to encode</param>
        /// <returns>Bitmap object</returns>
        public static Bitmap TextToDataMatrixBitmap( string text )
        {
            var arr = SerializeAndCompressString( text );
            return EncodeDataMatrix( arr );
        }

        /// <summary>
        /// Encodes a string as a DataMatrix Bitmap object and returns it as a raw Byte Array - useful in CrystalReports's BLOBs
        /// </summary>
        /// <param name="xml">String to encode</param>
        /// <returns>Byte Array of resulting barcode Bitmap</returns>
        public static byte[] XmlToBarcode( string xml )
        {
            var barcode = TextToDataMatrixBitmap( xml );

            return BmpToBytes_MemStream( barcode );
        }

        #endregion

        #region Deserialize & Decompress

        /// <summary>
        /// Decompress a given Byte Array and then deserialize it as String
        /// </summary>
        /// <param name="input">Byte Array to decompress and deserialize</param>
        /// <param name="ziptype">Compression type</param>
        /// <returns>Original String</returns>
        public static string DecompressAndDeserializeString( byte[] input, ZipUtils.ZipMethod ziptype )
        {
            var output = ZipUtils.UnzipArray( input, ziptype );

            // if no Binary Serialization is used ( no extra info added to the original XML ), deserialize with UTF8 encoding
            if( output.Length > 0 && output[0] == Encoding.UTF8.GetBytes( "<" )[0] )
            {
                return Encoding.UTF8.GetString( output );
            }

            #region ignore this code (legacy support)
            // else, use the obsolete approach, with Binary Deserialization
            var bf = new BinaryFormatter();
            using( var ms = new MemoryStream() )
            {
                ms.Write( output, 0, output.Length );
                ms.Seek( 0, 0 );
                return (string)bf.Deserialize( ms );
            }
            #endregion
        }

        #endregion
    }
}
