﻿#region Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// This file and its contents are protected by Romanian and International
// copyright laws. Unauthorized reproduction and/or distribution of all
// or any portion of the code contained herein is strictly prohibited
// and will result in severe civil and criminal penalties.
// Any violations of this copyright will be prosecuted
// to the fullest extent possible under law.
// ======================================================================
// THIS COPYRIGHT NOTICE MAY NOT BE REMOVED FROM THIS FILE.
// ======================================================================
#endregion

#region References
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO.Ports;
using System.Management;
using System.Timers;
#endregion

namespace Cnas.Siui.BarCode
{
    /// <summary>
    /// Singleton class to interface serial barcode readers
    /// </summary>
    public sealed class SerialPortReader
    {
        #region Fields

        private static volatile SerialPortReader instance;        // singleton instance
        private static readonly object syncObj = new object();   // mutual-exclusion
        private static SerialPort serialPort;

        private readonly Timer idleWatch = new Timer( 500 );    // after complete message received delay

        private readonly byte[] messagePrefixBZip2 = new byte[] { 0x42, 0x5A, 0x68, 0x39, 0x31, 0x41, 0x59, 0x26, 0x53, 0x59 }; // BZIP2 header
        private readonly byte[] messageSuffixBZip2 = new byte[] {}; // { 0x0D, 0x0A }; // BZIP2 ending (not used)

        private readonly byte[] messagePrefixJavaZip = new byte[] { 0x50, 0x4B, 0x03, 0x04 }; // JAVAZIP header
        private readonly byte[] messageSuffixJavaZip = new byte[] {}; // { 0x01, 0x00, 0x00, 0x00, 0x00 }; // JAVAZIP ending (not used)

        private readonly List<byte> currentTempMessage = new List<byte>();

        #endregion

        #region Events
        
        public delegate void MessageReceivedEventHandler( object sender, MessageReveivedEventArgs e );
        // Event rasied when a valid message ( as a byte array with a known prefix / sufix ) 
        // is received from the barcode reader
        public event MessageReceivedEventHandler MessageReceived;

        /// <summary>
        /// EventArgs class to encapsulate MessageReceived event information
        /// </summary>
        public class MessageReveivedEventArgs : EventArgs
        {
            public byte[] RawData
            {
                get;
                private set;
            }
            public ZipUtils.ZipMethod Compression
            {
                get;
                private set;
            }

            public MessageReveivedEventArgs( byte[] rawData, ZipUtils.ZipMethod compression )
            {
                this.RawData = rawData;
                this.Compression = compression;
            }
        }

        #endregion

        #region Constructor / Singleton instantiation

        private SerialPortReader()
        {
            Compression = ZipUtils.ZipMethod.Unknown; // default compression
            serialPort = new SerialPort(); // initialize the SerialPort object

            // attach event handlers
            serialPort.DataReceived += serialPort_DataReceived;
            idleWatch.Elapsed += IdleElapsedEvent;
        }

        public static SerialPortReader Instance
        {
            get
            {
                if( instance == null )
                {
                    lock( syncObj )
                    {
                        if( instance == null )
                            instance = new SerialPortReader();
                    }
                }

                return instance;
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// Flag to mark valid message
        /// </summary>
        public bool ValidMessageFound
        {
            get; private set; 
        }

        /// <summary>
        /// The latest valid received message from the barcode-reader
        /// </summary>
        public byte[] CurrentMessage 
        { 
            get; 
            private set; 
        }

        /// <summary>
        /// Compression type detected for the latest valid message 
        /// </summary>
        public ZipUtils.ZipMethod Compression
        {
            get; private set; 
        }

        /// <summary>
        /// Masks underlaying IsOpen property
        /// </summary>
        public bool IsSerialConnetionOpen
        {
            get 
            { 
                return serialPort.IsOpen; 
            }
        }

        #endregion

        #region Event Handlers

        /// <summary>
        /// Fired by the SerialPort object every time a minimum of ReceivedBytesThreshold bytes are received
        /// </summary>
        /// <param name="sender">SerialPort object</param>
        /// <param name="e">Event args</param>
        private void serialPort_DataReceived( object sender, SerialDataReceivedEventArgs e )
        {
            if( e.EventType != SerialData.Chars )
                return;

            idleWatch.Stop();

            var buffer = new byte[4069];
            var length = serialPort.Read( buffer, 0, buffer.Length );

            // add to list any received data chunks
            currentTempMessage.AddRange( buffer.Take( length ) );

            idleWatch.Start();
        }

        private bool CheckValidMessage( byte[] messagePrefix, byte[] messageSuffix )
        {
            if( messagePrefix.Length + messageSuffix.Length < currentTempMessage.Count )
            {
                // validate against the prefix
                if (messagePrefix.Where((t, i) => currentTempMessage[i] != t).Any())
                {
                    return false;
                }
                // check for suffix
                for (int i = 0; i < messageSuffix.Length; i++)
                {
                    if( currentTempMessage[currentTempMessage.Count - i - 1] != messageSuffix[messageSuffix.Length - i - 1] )
                    {
                        return false;
                    }
                }
            }
            else
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Fired after a complete message received or the idle time elapsed; 
        /// the message is validated against the defined prefix / suffix.
        /// </summary>
        /// <param name="source">Timer object</param>
        /// <param name="e">Event args</param>
        private void IdleElapsedEvent( object source, ElapsedEventArgs e )
        {
            idleWatch.Stop();

            this.ValidMessageFound = false;
            this.Compression = ZipUtils.ZipMethod.Unknown;

            if( this.CheckValidMessage( messagePrefixJavaZip, messageSuffixJavaZip ) )
            {
                ValidMessageFound = true;
                Compression = ZipUtils.ZipMethod.JavaZip;
            }
            else if( this.CheckValidMessage( messagePrefixBZip2, messageSuffixBZip2 ) )
            {
                ValidMessageFound = true;
                Compression = ZipUtils.ZipMethod.BZip2;
            }
            
            if( this.ValidMessageFound )
            {
                this.CurrentMessage = currentTempMessage.ToArray();

                if( this.MessageReceived != null )
                {
                    // rise the MessageReceived event with the CurrentMessage as args
                    this.MessageReceived( this, new MessageReveivedEventArgs( this.CurrentMessage, this.Compression ) );
                }
            }
            else
            {
                this.CurrentMessage = new byte[]{};
            }

            // cleanup the buffer
            currentTempMessage.Clear();
        }

        /// <summary>
        /// Opens a given serial port to listen for messages
        /// </summary>
        /// <param name="portName">Port name ("COM1")</param>
        /// <param name="baudRate">Baud rate (9600)</param>
        /// <param name="parity">Parity (None)</param>
        /// <param name="stopBits">Stop bits (1)</param>
        /// <param name="dataBits">Data bist (8)</param>
        /// <param name="handshake">Handshake protocol (None)</param>
        /// <returns>Connection state</returns>
        public bool StartListening( string portName = "COM1", int baudRate = 9600, Parity parity = Parity.None,
            StopBits stopBits = StopBits.One, int dataBits = 8, Handshake handshake = Handshake.None )
        {
            if( serialPort.IsOpen )
                serialPort.Close();

            serialPort.PortName = portName;
            serialPort.BaudRate = baudRate;
            serialPort.Parity = parity;
            serialPort.StopBits = stopBits;
            serialPort.DataBits = dataBits;
            serialPort.Handshake = handshake;
            //serialPort.DiscardNull = true;
            
            this.ValidMessageFound = false;

            try
            {
                serialPort.Open();
            }
            catch (Exception)
            {
                //throw;
            }

            return serialPort.IsOpen;
        }

        /// <summary>
        /// Close current serial port
        /// </summary>
        /// <returns>Connection state</returns>
        public bool StopListening()
        {
            if( serialPort.IsOpen )
                serialPort.Close();

            return serialPort.IsOpen;
        }

        #endregion

        #region Utilities

        /// <summary>
        /// Search and populate a list of clear product-name COM ports
        /// </summary>
        /// <returns>SerialPortName array of the found COM ports</returns>
        public SerialPortName[] ListSerialPorts()
        {
            const string query = "SELECT * FROM Win32_PnPEntity WHERE ConfigManagerErrorCode = 0";
            using( var searcher = new ManagementObjectSearcher( query ) )
            {
                string[] portnames = SerialPort.GetPortNames();
                var ports = searcher.Get().Cast<ManagementBaseObject>().ToList();

                // build array with all found captions
                var arrFound = ( from e in ports
                                 select e["Caption"].ToString()
                                 into caption
                                 from p in portnames
                                 where caption.Contains( string.Format( "({0})", p ) )
                                 orderby p
                                 select new SerialPortName
                                            {
                                                Caption = caption,
                                                Name = p
                                            }
                               ).ToArray();

                // append to the array missing ports ( that is ports that we didn't found in Win32_PnPEntity )
                var arrMissing = ( from p in portnames
                                   where p.ContainsAny( arrFound.Select( e => e.Name ).ToArray() ) == false
                                   select new SerialPortName
                                              {
                                                  Caption = p,
                                                  Name = p
                                              }
                                 ).ToArray();

                var arrAll = arrMissing.Concat( arrFound ).ToArray();

                return arrAll;
            }
        }

        public int[] ListBaudRates()
        {
            return new int[]{ 4800, 9600, 19200, 38400, 57600, 115200, 230400 };
        }
      
        public Parity[] ListParities()
        {
            return (Parity[])Enum.GetValues( typeof( Parity ) );
        }

        public StopBits[] ListStopBits()
        {
            return (StopBits[]) Enum.GetValues( typeof (StopBits) );
        }
        
        #endregion
    }
}
