﻿#region Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// This file and its contents are protected by Romanian and International
// copyright laws. Unauthorized reproduction and/or distribution of all
// or any portion of the code contained herein is strictly prohibited
// and will result in severe civil and criminal penalties.
// Any violations of this copyright will be prosecuted
// to the fullest extent possible under law.
// ======================================================================
// THIS COPYRIGHT NOTICE MAY NOT BE REMOVED FROM THIS FILE.
// ======================================================================
#endregion

#region References
using System;
using System.IO;
using ICSharpCode.SharpZipLib.BZip2;
using ICSharpCode.SharpZipLib.GZip;
using ICSharpCode.SharpZipLib.Zip;
#endregion

namespace Cnas.Siui.BarCode
{
    public class ZipUtils
    {
        /// <summary>
        /// ZipMethod values
        /// </summary>
        public enum ZipMethod
        {
            Unknown,
            /// <summary>
            /// GnuZip
            /// </summary>
            GnuZip,
            /// <summary>
            /// JavaZip
            /// </summary>
            JavaZip,
            /// <summary>
            /// BZip2
            /// </summary>
            BZip2,
            /// <summary>
            /// Gzip
            /// </summary>
            GZip
        }

        /// <summary>
        /// Compresses a byte array
        /// </summary>
        /// <param name="inputArray">Input array</param>
        /// <param name="packingMethod">Compression method</param>
        /// <returns>Compressed byte array</returns>
        public static byte[] ZipArray( byte[] inputArray, ZipMethod packingMethod = ZipMethod.JavaZip )
        {
            if( inputArray == null )
            {
                throw new ArgumentNullException( "inputArray" );
            }

            var buffer = new byte[Environment.Is64BitProcess ? 4096 : 2048];

            switch( packingMethod )
            {
                case ZipMethod.JavaZip:
                    #region JavaZip
                    try
                    {
                        using( var outStream = new MemoryStream() )
                        {
                            using( var inStream = new MemoryStream( inputArray ) )
                            {
                                using( var zip = new ZipOutputStream( outStream ) )
                                {
                                    int outSize = 0;
                                    zip.SetLevel( 9 );
                                    var entry = new ZipEntry( "array" )
                                    {
                                        DateTime = DateTime.Now
                                    };

                                    zip.PutNextEntry( entry );
                                    while( ( outSize = inStream.Read( buffer, 0, buffer.Length ) ) > 0 )
                                    {
                                        zip.Write( buffer, 0, outSize );
                                    }
                                }
                            }
                            return outStream.ToArray();
                        }
                    }
                    catch( Exception ex )
                    {
                        throw new ZipException( string.Format( Properties.Resources.ErrorZippingArray, "JavaZip", ex.Message ) );
                    }
                    #endregion
                case ZipMethod.BZip2:
                    #region BZip2
                    try
                    {
                        using( var outStream = new MemoryStream() )
                        {
                            using( var inStream = new MemoryStream( inputArray ) )
                            {
                                using( var zip = new BZip2OutputStream( outStream ) )
                                {
                                    int outSize = 0;
                                    while( ( outSize = inStream.Read( buffer, 0, buffer.Length ) ) > 0 )
                                    {
                                        zip.Write( buffer, 0, outSize );
                                    }
                                }
                            }
                            return outStream.ToArray();
                        }
                    }
                    catch( Exception ex )
                    {
                        throw new ZipException( string.Format( Properties.Resources.ErrorZippingArray, "BZip2", ex.Message ) );
                    }
                    #endregion
                case ZipMethod.GZip:
                    #region GZip
                    try
                    {
                        using( var outStream = new MemoryStream() )
                        {
                            using( var inStream = new MemoryStream( inputArray ) )
                            {
                                using( var zip = new GZipOutputStream( outStream ) )
                                {
                                    int outSize = 0;
                                    while( ( outSize = inStream.Read( buffer, 0, buffer.Length ) ) > 0 )
                                    {
                                        zip.Write( buffer, 0, outSize );
                                    }
                                }
                            }
                            return outStream.ToArray();
                        }
                    }
                    catch( Exception ex )
                    {
                        throw new ZipException( string.Format( Properties.Resources.ErrorZippingArray, "GZip", ex.Message ) );
                    }
                    #endregion
                default:
                    throw new NotImplementedException( packingMethod.ToString() );
            }
        }

        /// <summary>
        /// Decompresses a byte array
        /// </summary>
        /// <param name="inputArray">Input array</param>
        /// <param name="packingMethod">Compression method</param>
        /// <returns>Decompressed byte array</returns>
        public static byte[] UnzipArray( byte[] inputArray, ZipMethod packingMethod = ZipMethod.JavaZip )
        {
            if( inputArray == null )
            {
                throw new ArgumentNullException( "inputArray" );
            }

            var buffer = new byte[Environment.Is64BitProcess ? 4096 : 2048];

            switch( packingMethod )
            {
                case ZipMethod.JavaZip:
                    #region JavaZip
                    try
                    {
                        using( var outStream = new MemoryStream() )
                        {
                            using( var inStream = new MemoryStream( inputArray ) )
                            {
                                using( var zip = new ZipInputStream( inStream ) )
                                {
                                    // we're working with arrays, so we aspect to have only one entry
                                    if( zip.GetNextEntry() != null )
                                    {
                                        int outSize = 0;
                                        while( ( outSize = zip.Read( buffer, 0, buffer.Length ) ) > 0 )
                                        {
                                            outStream.Write( buffer, 0, outSize );
                                        }
                                    }
                                }
                            }
                            return outStream.ToArray();
                        }
                    }
                    catch( Exception ex )
                    {
                        throw new ZipException( string.Format( Properties.Resources.ErrorUnzippingArray, "JavaZip", ex.Message ) );
                    }
                    #endregion
                case ZipMethod.BZip2:
                    #region BZip2
                    try
                    {
                        using( var outStream = new MemoryStream() )
                        {
                            using( var inStream = new MemoryStream( inputArray ) )
                            {
                                using( var zip = new BZip2InputStream( inStream ) )
                                {
                                    int outSize = 0;
                                    while( ( outSize = zip.Read( buffer, 0, buffer.Length ) ) > 0 )
                                    {
                                        outStream.Write( buffer, 0, outSize );
                                    }
                                }
                            }
                            return outStream.ToArray();
                        }
                    }
                    catch( Exception ex )
                    {
                        throw new ZipException( string.Format( Properties.Resources.ErrorUnzippingArray, "BZip2", ex.Message ) );
                    }
                    #endregion
                case ZipMethod.GZip:
                    #region GZip
                    try
                    {
                        using( var outStream = new MemoryStream() )
                        {
                            using( var inStream = new MemoryStream( inputArray ) )
                            {
                                using( var zip = new GZipInputStream( inStream ) )
                                {
                                    int outSize = 0;
                                    while( ( outSize = zip.Read( buffer, 0, buffer.Length ) ) > 0 )
                                    {
                                        outStream.Write( buffer, 0, outSize );
                                    }
                                }
                            }
                            return outStream.ToArray();
                        }
                    }
                    catch( Exception ex )
                    {
                        throw new ZipException( string.Format( Properties.Resources.ErrorUnzippingArray, "GZip", ex.Message ) );
                    }
                    #endregion
                default:
                    throw new NotImplementedException( packingMethod.ToString() );
            }
        }
    }
}
