﻿#region Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// Copyright ©2011-2012, SIVECO Romania SA - All Rights Reserved
// ======================================================================
// This file and its contents are protected by Romanian and International
// copyright laws. Unauthorized reproduction and/or distribution of all
// or any portion of the code contained herein is strictly prohibited
// and will result in severe civil and criminal penalties.
// Any violations of this copyright will be prosecuted
// to the fullest extent possible under law.
// ======================================================================
// THIS COPYRIGHT NOTICE MAY NOT BE REMOVED FROM THIS FILE.
// ======================================================================
#endregion

#region References
using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using Cnas.Siui.BarCode;
using System.Drawing.Imaging;
using Cnas.Siui.BarCodeTest.Properties;
#endregion

namespace Cnas.Siui.BarCodeTest
{
    public partial class BarCodeTestForm : Form
    {
        #region Fields

        private bool _messageReceived = false;
        private bool _formClosing = false;

        #endregion

        #region Constructor

        public BarCodeTestForm()
        {
            InitializeComponent();
            LocalizeControls();
            PopulateControls();
        }

        #endregion

        #region Operations

        /// <summary>
        /// Fill captions from string resources
        /// </summary>
        private void LocalizeControls()
        {
            this.Text = Resources.FormTitle;
            groupBoxEncode.Text = Resources.GroupEncode;
            groupBoxDecode.Text = Resources.GroupDecode;
            buttonEncode.Text = Resources.BtnGenerate;
            buttonExport.Text = Resources.BtnExport;
            buttonScan.Text = Resources.BtnScan;
            buttonExit.Text = Resources.BtnExit;
            labelPort.Text = Resources.LabelPort;
            labelBaudRate.Text = Resources.LabelBaudRate;

            toolTip.SetToolTip( buttonEncode, Resources.ToolTipBtnGenerate );
            toolTip.SetToolTip( buttonExport, Resources.ToolTipBtnExport );
            toolTip.SetToolTip( buttonScan, Resources.ToolTipBtnScan );
            toolTip.SetToolTip( buttonExit, Resources.ToolTipBtnExit );
        }

        /// <summary>
        /// Fill controls with initial / persistent data
        /// </summary>
        private void PopulateControls()
        {
            #region COM port parameters

            comboBoxPort.DataSource = SerialPortReader.Instance.ListSerialPorts();
            comboBoxBaud.DataSource = SerialPortReader.Instance.ListBaudRates();
            comboBoxPort.DisplayMember = "Caption";
            comboBoxPort.ValueMember = "Name";
            
            // get previous working parameters to setup the serial connection
            var portNo = Properties.Settings.Default.SerialPortNumber;
            if( portNo != string.Empty && ( comboBoxPort.DataSource as SerialPortName[] ).Any( sp => sp.Name == portNo ) )
            {
                comboBoxPort.SelectedValue = portNo;
            }

            int baudRate = Properties.Settings.Default.SerialPortBaudRate;
            if( ( comboBoxBaud.DataSource as int[] ).Contains( baudRate ) )
            {
                comboBoxBaud.Text = baudRate.ToString();
            }

            #endregion

            // event to be raised when a valid message is received from the barcode reader
            SerialPortReader.Instance.MessageReceived += SerialPortReader_MessageReceived;

            // initialize button captions
            ToggleCaption();
        }

        private void ClearResults()
        {
            pbGenDataMatrix.Image = null;
            buttonExport.Enabled = false;
        }

        private void ToggleCaption()
        {
            if( backgroundWorker.IsBusy )
            {
                buttonScan.Text = Resources.BarCodeTestForm_ToggleCaption_Anuleaza;
                labelStatus.Text = Resources.BarCodeTestForm_ToggleCaption_CanReadBarcode;
                panelScanControls.Enabled = false;
                tbDecodedText.BackColor = Color.LightGray;
            }
            else
            {
                buttonScan.Text = Resources.BarCodeTestForm_ToggleCaption_Scan;
                panelScanControls.Enabled = true;
                tbDecodedText.BackColor = Color.White;
            }
        }

        /// <summary>
        /// Generate correspoding DataMatrix barcode
        /// </summary>
        private void EncodeTextAsDataMatrix()
        {
            var textToEncode = tbTextToEncode.Text.Trim();
            if( textToEncode.Length > 0 )
            {
                this.pbGenDataMatrix.Image = BarcodeUtils.TextToDataMatrixBitmap(textToEncode);
                buttonExport.Enabled = true;
            }
        }

        /// <summary>
        /// Start listening to the selected COM port for messages from the barcode reader,
        /// on a BackgroundWorker thread
        /// </summary>
        private void StartScanning()
        {
            var portNo = comboBoxPort.SelectedValue.ToString();
            var baudRate = (int)comboBoxBaud.SelectedValue;

            _messageReceived = false;
            tbDecodedText.Text = "";

            // open serial port communication
            if( SerialPortReader.Instance.StartListening( portNo, baudRate ) )
            {
                // if succedes, then save the parameters
                Settings.Default.SerialPortNumber = portNo;
                Settings.Default.SerialPortBaudRate = baudRate;
                Settings.Default.Save();
                
                // start listening
                if( this.backgroundWorker.IsBusy == false )
                {
                    this.backgroundWorker.RunWorkerAsync();                   
                }
            }
            else
            {
                labelStatus.Text = Resources.BarCodeTestForm_StartScanning_CannotOpenPort;
            }
            
            ToggleCaption();
        }

        /// <summary>
        /// Cancels scanning and its working thread
        /// </summary>
        private void StopScanning()
        {
            labelStatus.Text = Resources.BarCodeTestForm_StopScanning_StoppingScan;
            backgroundWorker.CancelAsync();
            this.Invoke( () => progressBarScan.Value = 0 );
        }
        
        /// <summary>
        /// Export DataMatrix barcode as an image file
        /// </summary>
        private void ExportDataMatrix()
        {
            if( this.pbGenDataMatrix.Image != null )
            {
                using( var dlg = new SaveFileDialog())
                {
                    dlg.Filter = "Jpeg|*.jpg|Bitmap|*.bmp|Tiff|*.tiff";
                    dlg.Title = Resources.BarCodeTestForm_ExportDataMatrix_ExportAsImage;
                    dlg.ShowDialog();
                    
                    if( dlg.FileName != "" )
                    {
                        // saves the Image via a FileStream created by the OpenFile method.
                        var fs = (System.IO.FileStream)dlg.OpenFile();
                       
                        switch( dlg.FilterIndex )
                        {
                            case 1:
                                pbGenDataMatrix.Image.Save( fs, ImageFormat.Jpeg );
                                break;

                            case 2:
                                pbGenDataMatrix.Image.Save( fs, ImageFormat.Bmp );
                                break;

                            case 3:
                                pbGenDataMatrix.Image.Save( fs, ImageFormat.Tiff );
                                break;
                        }

                        fs.Close();
                    }
                }
            }
        }

        #endregion

        #region Event Handlers

        private void buttonEncode_Click( object sender, EventArgs e )
        {
            EncodeTextAsDataMatrix();
        }

        private void BarCodeTestForm_Load( object sender, EventArgs e )
        {
            EncodeTextAsDataMatrix();
        }

        private void tbTextToEncode_TextChanged( object sender, EventArgs e )
        {
            ClearResults();
        }

        /// <summary>
        /// Event to handle receiving messages form the barcode reader
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void SerialPortReader_MessageReceived( object sender, SerialPortReader.MessageReveivedEventArgs e )
        {
            try
            {
                SerialPortReader.Instance.StopListening();
                if( SerialPortReader.Instance.ValidMessageFound )
                {
                    this.Invoke( () => this.tbDecodedText.Text = BarcodeUtils.DecompressAndDeserializeString( e.RawData, e.Compression ) );
                }
                else
                {
                    this.Invoke( () => this.tbDecodedText.Text = "" );
                }

                _messageReceived = true;
                StopScanning();
            }
            catch( Exception ex )
            {
                //var msg = ex.Message;
            }
            finally
            {
                _messageReceived = true;
                StopScanning();
            }
        }

        #region Worker

        [System.Diagnostics.DebuggerStepThrough()]
        private void backgroundWorker_DoWork( object sender, DoWorkEventArgs e )
        {
            // get worker data
            var worker = (BackgroundWorker)sender;
            if( worker.CancellationPending == true )
            {
                e.Cancel = true;
                return;
            }

            // setup thread
            Thread.CurrentThread.Priority = ThreadPriority.Lowest;

            // loop until receive a scan, or the user cancel-it
            int i = 0;
            while( _messageReceived == false
                && worker.CancellationPending == false )
            {
                worker.ReportProgress( i, "" );
                if( ++i == 100 )
                    i = 0;

                Thread.Sleep( 100 );
            }

            if( _messageReceived == false )
            {
                e.Cancel = true;
            }
        }

        private void backgroundWorker_ProgressChanged( object sender, ProgressChangedEventArgs e )
        {
            // update user state, if exists
            var userState = e.UserState as string;

            progressBarScan.Value = Math.Min( 100, e.ProgressPercentage );
        }

        private void backgroundWorker_RunWorkerCompleted( object sender, RunWorkerCompletedEventArgs e )
        {
            // close the serial port connection
            SerialPortReader.Instance.StopListening();
            ToggleCaption();
            labelStatus.Text = "";

            if( _formClosing )
            {
                Close();
            }
            
            if( e.Cancelled == true )
            {
                // diplay user cancelation message
                labelStatus.Text = "";
            }
            else if( e.Error == null )
            {
                // display success message
                labelStatus.Text = Resources.BarCodeTestForm_backgroundWorker_RunWorkerCompleted_ScanSuccess;
            }
            else
            {
                // display error message
                labelStatus.Text = string.Format( "Eroare: {0}", e.Error.Message);
            }
        }

        #endregion

        private void buttonScan_Click( object sender, EventArgs e )
        {
            if( backgroundWorker.IsBusy )
            {
                StopScanning();
            }
            else
            {
                StartScanning();
            }
        }

        private void BarCodeTestForm_FormClosing( object sender, FormClosingEventArgs e )
        {
            if( backgroundWorker.IsBusy )
            {
                StopScanning();
                e.Cancel = true;
                _formClosing = true;
            }
        }
        
        private void buttonExport_Click( object sender, EventArgs e )
        {
            ExportDataMatrix();
        }

        private void buttonExit_Click( object sender, EventArgs e )
        {
            Close();
        }

        #endregion
    }
}
